<?php

namespace YankeeToolkit\Widgets;

use Elementor\Controls_Manager;
use Elementor\Group_Control_Image_Size;
use Elementor\Group_Control_Typography;
use Elementor\Widget_Base;
use WP_Query;

// Exit if accessed directly
if ( !defined( 'ABSPATH' ) ) {
    exit;
}

class Yankee_Portfolio_Filter extends Widget_Base {

    public function get_name() {
        return 'yankee-portfolio-filter';
    }

    public function get_title() {
        return esc_html__( 'Portfolio Filter', 'yankee-toolkit' );
    }

    public function get_icon() {
        return 'eicon-posts-justified';
    }

    public function get_style_depends() {
        return ['yankee-toolkit'];
    }

    public function get_script_depends() {
        return ['imagesloaded', 'isotope', 'yankee-toolkit'];
    }

    public function get_keywords() {
        return ['yankee', 'portfolio', 'recent', 'project'];
    }

    public function get_categories() {
        return ['yankee_element'];
    }

    protected function _register_controls() {

        $this->start_controls_section(
            'widget_content',
            [
                'label' => esc_html__( 'Portfolio Query', 'yankee-toolkit' ),
            ]
        );

        $this->add_control(
            'portfolio_cats',
            [
                'label'       => esc_html__( 'Select Categories', 'yankee-toolkit' ),
                'type'        => Controls_Manager::SELECT2,
                'options'     => $this->select_param_cates_post(),
                'multiple'    => true,
                'label_block' => true,
                'placeholder' => esc_html__( 'All Categories', 'yankee-toolkit' ),
            ]
        );

        $this->add_control(
            'show_count', [
                'label'   => esc_html__( 'Limit Item', 'yankee-toolkit' ),
                'type'    => Controls_Manager::NUMBER,
                'default' => 10,
                'min'     => 1,
            ]
        );

        $this->add_control(
            'order_by', [
                'label'   => esc_html__( 'Order By', 'yankee-toolkit' ),
                'type'    => Controls_Manager::SELECT,
                'options' => [
                    'none'       => esc_html__( 'None', 'yankee-toolkit' ),
                    'ID'         => esc_html__( 'Post ID', 'yankee-toolkit' ),
                    'author'     => esc_html__( 'Post Author', 'yankee-toolkit' ),
                    'title'      => esc_html__( 'Title', 'yankee-toolkit' ),
                    'date'       => esc_html__( 'Date', 'yankee-toolkit' ),
                    'modified'   => esc_html__( 'Last Modified Date', 'yankee-toolkit' ),
                    'rand'       => esc_html__( 'Random', 'yankee-toolkit' ),
                    'menu_order' => esc_html__( 'Menu Order', 'yankee-toolkit' ),
                ],
                'default' => 'date',
            ]
        );

        $this->add_control(
            'short_by', [
                'label'   => esc_html__( 'Order', 'yankee-toolkit' ),
                'type'    => Controls_Manager::SELECT,
                'options' => [
                    'ASC'  => esc_html__( 'Ascending', 'yankee-toolkit' ),
                    'DESC' => esc_html__( 'Descending', 'yankee-toolkit' ),
                ],
                'default' => 'DESC',
            ]
        );

        $this->end_controls_section();

        $this->start_controls_section(
            'portfolio_settings',
            [
                'label' => esc_html__( 'Settings', 'yankee-toolkit' ),
            ]
        );

        $this->add_control(
            'portfolio_layout', [
                'label'   => esc_html__( 'Portfolio Layout', 'yankee-toolkit' ),
                'type'    => Controls_Manager::SELECT,
                'options' => [
                    'layout-grid'         => esc_html__( 'Grid Layout', 'yankee-toolkit' ),
                    'layout-masonary'     => esc_html__( 'Masonary Layout', 'yankee-toolkit' ),
                    'layout-masonary-two' => esc_html__( 'Masonary Layout Two', 'yankee-toolkit' ),
                ],
                'default' => 'layout-grid',
            ]
        );

        $this->add_group_control(
            Group_Control_Image_Size::get_type(),
            [
                'name'    => 'portfolio_thumbnail',
                'default' => 'full',
                'exclude' => [
                    'custom',
                ],
            ]
        );

        $this->add_control(
            'title_word',
            [
                'label'   => esc_html__( 'Title Length', 'yankee-toolkit' ),
                'type'    => Controls_Manager::NUMBER,
                'default' => 4,
            ]
        );

        $this->add_control(
            'show_filter',
            [
                'label'        => __( 'Show Filter', 'yankee-toolkit' ),
                'type'         => Controls_Manager::SWITCHER,
                'label_on'     => __( 'Show', 'yankee-toolkit' ),
                'label_off'    => __( 'Hide', 'yankee-toolkit' ),
                'return_value' => 'yes',
                'default'      => 'yes',
            ]
        );

        $this->add_control(
            'all_item_text', [
                'label'       => esc_html__( 'All Item Text', 'yankee-toolkit' ),
                'type'        => Controls_Manager::TEXT,
                'label_block' => true,
                'default'     => esc_html__( 'All Works', 'yankee-toolkit' ),
                'condition'   => [
                    'show_filter' => 'yes',
                ],
            ]
        );

        $this->add_control(
            'show_cat',
            [
                'label'        => __( 'Show Categories', 'yankee-toolkit' ),
                'type'         => Controls_Manager::SWITCHER,
                'label_on'     => __( 'Show', 'yankee-toolkit' ),
                'label_off'    => __( 'Hide', 'yankee-toolkit' ),
                'return_value' => 'yes',
                'default'      => 'yes',
            ]
        );

        $this->add_control(
            'column_heading',
            [
                'label'     => esc_html__( 'Column', 'yankee-toolkit' ),
                'type'      => Controls_Manager::HEADING,
                'separator' => 'before',
                'condition' => [
                    'portfolio_layout' => 'layout-grid',
                ],
            ]
        );

        $this->add_control(
            'desktop_col',
            [
                'label'     => esc_html__( 'Desktop', 'yankee-toolkit' ),
                'type'      => Controls_Manager::SELECT,
                'default'   => 'col-lg-4',
                'options'   => [
                    'col-lg-12' => esc_html__( '1 column', 'yankee-toolkit' ),
                    'col-lg-6'  => esc_html__( '2 column', 'yankee-toolkit' ),
                    'col-lg-4'  => esc_html__( '3 column', 'yankee-toolkit' ),
                    'col-lg-3'  => esc_html__( '4 column', 'yankee-toolkit' ),
                ],
                'condition' => [
                    'portfolio_layout' => 'layout-grid',
                ],
            ]
        );

        $this->add_control(
            'tablet_col',
            [
                'label'     => esc_html__( 'Tablet', 'yankee-toolkit' ),
                'type'      => Controls_Manager::SELECT,
                'default'   => 'col-md-6',
                'options'   => [
                    'col-md-12' => esc_html__( '1 column', 'yankee-toolkit' ),
                    'col-md-6'  => esc_html__( '2 column', 'yankee-toolkit' ),
                    'col-md-4'  => esc_html__( '3 column', 'yankee-toolkit' ),
                    'col-md-3'  => esc_html__( '4 column', 'yankee-toolkit' ),
                ],
                'condition' => [
                    'portfolio_layout' => 'layout-grid',
                ],
            ]
        );

        $this->add_control(
            'mobile_col',
            [
                'label'     => esc_html__( 'Mobile', 'yankee-toolkit' ),
                'type'      => Controls_Manager::SELECT,
                'default'   => 'col-sm-6',
                'options'   => [
                    'col-sm-12' => esc_html__( '1 column', 'yankee-toolkit' ),
                    'col-sm-6'  => esc_html__( '2 column', 'yankee-toolkit' ),
                    'col-sm-4'  => esc_html__( '3 column', 'yankee-toolkit' ),
                ],
                'condition' => [
                    'portfolio_layout' => 'layout-grid',
                ],
            ]
        );

        $this->add_control(
            'mobile_sm_col',
            [
                'label'     => esc_html__( 'Mobile(Small)', 'yankee-toolkit' ),
                'type'      => Controls_Manager::SELECT,
                'default'   => 'col-12',
                'options'   => [
                    'col-12' => esc_html__( '1 column', 'yankee-toolkit' ),
                    'col-6'  => esc_html__( '2 column', 'yankee-toolkit' ),
                    'col-4'  => esc_html__( '3 column', 'yankee-toolkit' ),
                ],
                'condition' => [
                    'portfolio_layout' => 'layout-grid',
                ],
            ]
        );

        $this->end_controls_section();

        $this->start_controls_section(
            'widget_nav_style',
            [
                'label'     => esc_html__( 'Portfolio Nav', 'yankee-toolkit' ),
                'tab'       => Controls_Manager::TAB_STYLE,
                'condition' => [
                    'show_filter' => 'yes',
                ],
            ]
        );

        $this->add_responsive_control(
            'column_align', [
                'label'     => esc_html__( 'Nav Btn Alignment', 'yankee-toolkit' ),
                'type'      => Controls_Manager::SELECT,
                'options'   => [
                    'flex-start' => esc_html__( 'Left', 'yankee-toolkit' ),
                    'center'     => esc_html__( 'Center', 'yankee-toolkit' ),
                    'flex-end'   => esc_html__( 'End', 'yankee-toolkit' ),
                ],
                'selectors' => [
                    '{{WRAPPER}} .portfolio-nav ul' => 'justify-content: {{VALUE}};',
                ],
            ]
        );

        $this->add_responsive_control(
            'nav_area_margin',
            [
                'label'      => esc_html__( 'Nav Area Margin', 'yankee-toolkit' ),
                'type'       => Controls_Manager::DIMENSIONS,
                'size_units' => ['px', 'em', '%'],
                'selectors'  => [
                    '{{WRAPPER}} .portfolio-nav' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->add_control(
            'nav_item_heading',
            [
                'label'     => esc_html__( 'Nav Items', 'yankee-toolkit' ),
                'type'      => Controls_Manager::HEADING,
                'separator' => 'before',
            ]
        );

        $this->add_responsive_control(
            'nav_item_margin',
            [
                'label'      => esc_html__( 'Margin', 'yankee-toolkit' ),
                'type'       => Controls_Manager::DIMENSIONS,
                'size_units' => ['px', 'em', '%'],
                'selectors'  => [
                    '{{WRAPPER}} .portfolio-nav ul li'         => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                    '{{WRAPPER}} .portfolio-nav ul li::before' => 'left: calc( -{{LEFT}}{{UNIT}} / 2 );',
                ],
            ]
        );

        $this->add_responsive_control(
            'nav_item_padding',
            [
                'label'      => esc_html__( 'Padding', 'yankee-toolkit' ),
                'type'       => Controls_Manager::DIMENSIONS,
                'size_units' => ['px', 'em', '%'],
                'selectors'  => [
                    '{{WRAPPER}} .portfolio-nav ul li' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name'     => 'nav_item_typography',
                'selector' => '{{WRAPPER}} .portfolio-nav ul li',
            ]
        );

        $this->start_controls_tabs( 'nav_items' );

        $this->start_controls_tab(
            'nav_item_normal',
            ['label' => esc_html__( 'Normal', 'yankee-toolkit' )]
        );

        $this->add_control(
            'nav_item_normal_bg_color',
            [
                'label'     => esc_html__( 'Background', 'yankee-toolkit' ),
                'type'      => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .portfolio-nav ul li' => 'background-color: {{VALUE}};',
                ],
            ]
        );

        $this->add_control(
            'nav_item_normal_color',
            [
                'label'     => esc_html__( 'Color', 'yankee-toolkit' ),
                'type'      => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .portfolio-nav ul li' => 'color: {{VALUE}};',
                ],
            ]
        );

        $this->end_controls_tab();

        $this->start_controls_tab(
            'nav_item_hover',
            ['label' => esc_html__( 'Hover', 'yankee-toolkit' )]
        );

        $this->add_control(
            'nav_item_hover_bg_color',
            [
                'label'     => esc_html__( 'Background', 'yankee-toolkit' ),
                'type'      => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .portfolio-nav ul li:hover, {{WRAPPER}} .portfolio-nav ul li.active' => 'background-color: {{VALUE}};',
                    '{{WRAPPER}} .portfolio-nav ul li::before'                                        => 'color: {{VALUE}};',
                ],
            ]
        );

        $this->add_control(
            'nav_item_hover_color',
            [
                'label'     => esc_html__( 'Color', 'yankee-toolkit' ),
                'type'      => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .portfolio-nav ul li:hover, {{WRAPPER}} .portfolio-nav ul li.active' => 'color: {{VALUE}};',
                ],
            ]
        );

        $this->end_controls_tab();

        $this->end_controls_tabs();

        $this->end_controls_section();

        $this->start_controls_section(
            'widget_items_style',
            [
                'label' => esc_html__( 'Portfolio Items', 'yankee-toolkit' ),
                'tab'   => Controls_Manager::TAB_STYLE,
            ]
        );

        $this->add_responsive_control(
            'items_grid_gap',
            [
                'label'       => esc_html__( 'Grid Gap', 'yankee-toolkit' ),
                'type'        => Controls_Manager::SLIDER,
                'size_units'  => ['px'],
                'range'       => [
                    'px' => [
                        'min'  => 0,
                        'max'  => 50,
                        'step' => 1,
                    ],
                ],
                'selectors'   => [
                    '{{WRAPPER}} .portfolio-items.row'            => 'margin: -{{SIZE}}{{UNIT}}',
                    '{{WRAPPER}} .portfolio-items.row .grid-item' => 'padding: {{SIZE}}{{UNIT}};',
                ],
                'description' => esc_html__( 'You may have to reload this to see the change of grid-gap properly', 'yankee-toolkit' ),
            ]
        );

        $this->add_control(
            'content_area_heading',
            [
                'label'     => esc_html__( 'Content Area', 'yankee-toolkit' ),
                'type'      => Controls_Manager::HEADING,
                'separator' => 'before',
            ]
        );

        $this->add_responsive_control(
            'content_area_padding',
            [
                'label'      => esc_html__( 'Padding', 'yankee-toolkit' ),
                'type'       => Controls_Manager::DIMENSIONS,
                'size_units' => ['px', 'em', '%'],
                'selectors'  => [
                    '{{WRAPPER}} .portfolio-items .portfolio-item .portfolio-desc' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->add_control(
            'content_area_bg',
            [
                'label'     => esc_html__( 'Background', 'yankee-toolkit' ),
                'type'      => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .portfolio-items .portfolio-item .portfolio-desc' => 'background-color: {{VALUE}};',
                ],
            ]
        );

        $this->add_control(
            'title_haeding',
            [
                'label'     => esc_html__( 'Title', 'yankee-toolkit' ),
                'type'      => Controls_Manager::HEADING,
                'separator' => 'before',
            ]
        );

        $this->add_control(
            'title_color',
            [
                'label'     => esc_html__( 'Color', 'yankee-toolkit' ),
                'type'      => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .portfolio-items .portfolio-item .portfolio-desc .title a' => 'color: {{VALUE}};',
                ],
            ]
        );

        $this->add_control(
            'title__hover_color',
            [
                'label'     => esc_html__( 'Hover Color', 'yankee-toolkit' ),
                'type'      => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .portfolio-items .portfolio-item .portfolio-desc .title a:hover' => 'color: {{VALUE}};',
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name'     => 'title_typography',
                'selector' => '{{WRAPPER}} .portfolio-items .portfolio-item .portfolio-desc .title',
            ]
        );

        $this->add_control(
            'categories_title',
            [
                'label'     => esc_html__( 'Categories', 'yankee-toolkit' ),
                'type'      => Controls_Manager::HEADING,
                'separator' => 'before',
                'condition' => [
                    'show_cat' => 'yes',
                ],
            ]
        );

        $this->add_control(
            'categories_color',
            [
                'label'     => esc_html__( 'Color', 'yankee-toolkit' ),
                'type'      => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .portfolio-items .portfolio-item .portfolio-desc .cat' => 'color: {{VALUE}};',
                ],
                'condition' => [
                    'show_cat' => 'yes',
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name'      => 'categories_typography',
                'selector'  => '{{WRAPPER}} .portfolio-items .portfolio-item .portfolio-desc .cat',
                'condition' => [
                    'show_cat' => 'yes',
                ],
            ]
        );

        $this->end_controls_section();
    }

    protected function render() {
        $settings = $this->get_settings();
        $column_edit = 1;

        if( $settings['portfolio_layout'] == 'layout-masonary' ) {
            $grid_sizer = 'grid-sizer col-lg-3 col-sm-6';
        } elseif( $settings['portfolio_layout'] == 'layout-masonary-two' ) {
            $grid_sizer = 'grid-sizer col-lg-4 col-sm-6';
        } else {
            $grid_sizer = 'grid-sizer ' . $settings['desktop_col'] .' '. $settings['tablet_col'] .' '. $settings['mobile_col'] .' '. $settings['mobile_sm_col'];
        }
        ?>
            <div class="yankee-portfolio-widgets">
                <?php if( $settings['show_filter'] == 'yes' ) : ?>
                    <div class="portfolio-nav clearfix">
                        <ul>
                            <li data-filter="*" class="active"><?php echo esc_html( $settings['all_item_text'] ) ?></li>
                            <?php
                                if ( $settings['portfolio_cats'] ) {
                                    $categories = $settings['portfolio_cats'];

                                    foreach( (array)$categories as $categorie ) {
                                        $cates = get_term_by( 'slug', $categorie, 'portfolios_cat' );
                                        $cat_name = $cates->name;
                                        $cat_slug = $cates->slug;

                                        printf( '<li data-filter=".%1$s">%2$s</li>',
                                            esc_attr( $cat_slug ),
                                            esc_html( $cat_name )
                                        );
                                    }
                                } else {
                                    $categories = get_terms( 'portfolios_cat' );

                                    foreach( (array)$categories as $categorie ) {
                                        $cat_name = $categorie->name;
                                        $cat_slug = $categorie->slug;

                                        printf( '<li data-filter=".%1$s">%2$s</li>',
                                            esc_attr( $cat_slug ),
                                            esc_html( $cat_name )
                                        );
                                    }
                                }
                            ?>
                        </ul>
                    </div>
                <?php endif; ?>

                <div class="row portfolio-items portfolio-filter portfolio-<?php echo esc_attr( $settings['portfolio_layout'] ) ?>">
                    <div class="<?php echo esc_attr( $grid_sizer ) ?>"></div>
                    <?php
                        if ( $settings['portfolio_cats'] ) {
                            $args = array(
                                'post_type'      => 'portfolios',
                                'post_status'    => 'publish',
                                'posts_per_page' => $settings['show_count'],
                                'orderby'        => $settings['order_by'],
                                'order'          => $settings['short_by'],
                                'ignore_sticky_posts' => 1,
                                'tax_query' => array(
                                    array(
                                        'taxonomy' => 'portfolios_cat',
                                        'field'    => 'slug',
                                        'terms'    => $settings['portfolio_cats'],
                                    ),
                                ),
                            );
                        } else {
                            $args = array(
                                'post_type'      => 'portfolios',
                                'post_status'    => 'publish',
                                'posts_per_page' => $settings['show_count'],
                                'orderby'        => $settings['order_by'],
                                'order'          => $settings['short_by'],
                                'ignore_sticky_posts' => 1,
                            );
                        }

                        $wp_query = new WP_Query($args);
                        while ( $wp_query -> have_posts() ) : $wp_query -> the_post();
                            $idd = get_the_ID();
                            $portfolio_categorys = get_the_terms( $idd, 'portfolios_cat' );
                            $item_cat_slug = '';
                            $item_cat_name = '';

                            if ($portfolio_categorys && ! is_wp_error( $portfolio_categorys) ){
                                $cat_list_slug = array();
                                $cat_list_name = array();

                                foreach ($portfolio_categorys as $category) {
                                    $cat_list_slug[] = $category->slug;
                                    $cat_list_name[] = $category->name;
                                }

                                $item_cat_slug = join( " ", $cat_list_slug );
                                $item_cat_name = join( ", ", $cat_list_name );
                            }

                            if ( $settings['title_word'] ) {
                                $the_title = wp_trim_words(get_the_title(), $settings['title_word']);
                            } else {
                                $the_title = get_the_title();
                            }

                            $classes = 'grid-item ' . $settings['desktop_col'] .' '. $settings['tablet_col'] .' '. $settings['mobile_col'] .' '. $settings['mobile_sm_col'] .' '.$item_cat_slug;

                            if( $settings['portfolio_layout'] == 'layout-masonary' ) {
                                if ($column_edit > 5){
                                    $column_edit = 1;
                                }

                                $classes = 'grid-item col-lg-3 col-sm-6' .' '.$item_cat_slug;

                                if ( $column_edit == 2 ){
                                    $classes = 'grid-item grid-big-item col-lg-6 col-sm-6' .' '.$item_cat_slug;
                                }
                            }

                            if( $settings['portfolio_layout'] == 'layout-masonary-two' ) {
                                if ($column_edit > 7){
                                    $column_edit = 1;
                                }

                                $classes = 'grid-item col-lg-4 col-sm-6' .' '.$item_cat_slug;

                                if ( $column_edit == 4 || $column_edit == 7 ){
                                    $classes = 'grid-item grid-wide-item col-lg-8 col-sm-6' .' '.$item_cat_slug;
                                }
                            }

                            $column_edit++;
                        ?>

                        <div class="<?php echo esc_attr( $classes ) ?>">
                            <div class="portfolio-item">
                                <div class="portfolio-thumb">
                                    <div class="thumb" style="background-image: url(<?php echo esc_url( get_the_post_thumbnail_url( $idd, $settings['portfolio_thumbnail_size'] ) ) ?>);"></div>
                                </div>
                                <div class="portfolio-desc">
                                    <?php if( !empty( $item_cat_name ) && $settings['show_cat'] ) : ?>
                                        <span class="cat"><?php echo esc_html( $item_cat_name ) ?></span>
                                    <?php endif; ?>
                                    <h5 class="title"><a href="<?php echo esc_url( get_the_permalink() ) ?>"><?php echo esc_html( $the_title ) ?></a></h5>
                                </div>
                            </div>
                        </div>

                        <?php endwhile;
                        wp_reset_postdata();
                    ?>
                </div>
            </div>
        <?php
    }

    protected function select_param_cates_post() {
        $category = get_terms( 'portfolios_cat' );
        $cat = [];
        foreach ( $category as $item ) {
            if ( $item ) {
                $cat[$item->slug] = $item->name;
            }
        }
        return $cat;
    }
}