<?php

namespace YankeeToolkit\Widgets;

use Elementor\Controls_Manager;
use Elementor\Group_Control_Border;
use Elementor\Group_Control_Image_Size;
use Elementor\Group_Control_Typography;
use Elementor\Widget_Base;
use WP_Query;

// Exit if accessed directly
if ( !defined( 'ABSPATH' ) ) {
    exit;
}

class Yankee_Portfolio_List extends Widget_Base {

    public function get_name() {
        return 'yankee-portfolio-list';
    }

    public function get_title() {
        return esc_html__( 'Portfolio List', 'yankee-toolkit' );
    }

    public function get_icon() {
        return 'eicon-post-list';
    }

    public function get_style_depends() {
        return ['yankee-toolkit'];
    }

    public function get_keywords() {
        return ['yankee', 'portfolio', 'recent', 'project'];
    }

    public function get_categories() {
        return ['yankee_element'];
    }

    protected function _register_controls() {

        $this->start_controls_section(
            'widget_content',
            [
                'label' => esc_html__( 'Portfolio Query', 'yankee-toolkit' ),
            ]
        );

        $this->add_control(
            'portfolio_cats',
            [
                'label'       => esc_html__( 'Select Categories', 'yankee-toolkit' ),
                'type'        => Controls_Manager::SELECT2,
                'options'     => $this->select_param_cates_post(),
                'multiple'    => true,
                'label_block' => true,
                'placeholder' => esc_html__( 'All Categories', 'yankee-toolkit' ),
            ]
        );

        $this->add_control(
            'show_count', [
                'label'   => esc_html__( 'Limit Item', 'yankee-toolkit' ),
                'type'    => Controls_Manager::NUMBER,
                'default' => 10,
                'min'     => 1,
            ]
        );

        $this->add_control(
            'order_by', [
                'label'   => esc_html__( 'Order By', 'yankee-toolkit' ),
                'type'    => Controls_Manager::SELECT,
                'options' => [
                    'none'       => esc_html__( 'None', 'yankee-toolkit' ),
                    'ID'         => esc_html__( 'Post ID', 'yankee-toolkit' ),
                    'author'     => esc_html__( 'Post Author', 'yankee-toolkit' ),
                    'title'      => esc_html__( 'Title', 'yankee-toolkit' ),
                    'date'       => esc_html__( 'Date', 'yankee-toolkit' ),
                    'modified'   => esc_html__( 'Last Modified Date', 'yankee-toolkit' ),
                    'rand'       => esc_html__( 'Random', 'yankee-toolkit' ),
                    'menu_order' => esc_html__( 'Menu Order', 'yankee-toolkit' ),
                ],
                'default' => 'date',
            ]
        );

        $this->add_control(
            'short_by', [
                'label'   => esc_html__( 'Order', 'yankee-toolkit' ),
                'type'    => Controls_Manager::SELECT,
                'options' => [
                    'ASC'  => esc_html__( 'Ascending', 'yankee-toolkit' ),
                    'DESC' => esc_html__( 'Descending', 'yankee-toolkit' ),
                ],
                'default' => 'DESC',
            ]
        );

        $this->end_controls_section();

        $this->start_controls_section(
            'portfolio_settings',
            [
                'label' => esc_html__( 'Settings', 'yankee-toolkit' ),
            ]
        );

        $this->add_group_control(
            Group_Control_Image_Size::get_type(),
            [
                'name'    => 'portfolio_thumbnail',
                'default' => 'full',
                'exclude' => [
                    'custom',
                ],
            ]
        );

        $this->add_control(
            'title_word',
            [
                'label'   => esc_html__( 'Title Length', 'yankee-toolkit' ),
                'type'    => Controls_Manager::NUMBER,
                'default' => 6,
            ]
        );

        $this->add_control(
            'content_word',
            [
                'label'   => esc_html__( 'Excerpt Length', 'yankee-toolkit' ),
                'type'    => Controls_Manager::NUMBER,
                'default' => 20,
            ]
        );

        $this->add_control(
            'button_text', [
                'label'       => esc_html__( 'Button Text', 'yankee-toolkit' ),
                'type'        => Controls_Manager::TEXT,
                'label_block' => true,
                'default'     => esc_html__( 'Project Details', 'yankee-toolkit' ),
            ]
        );

        $this->add_control(
            'show_cat',
            [
                'label'        => __( 'Show Categories', 'yankee-toolkit' ),
                'type'         => Controls_Manager::SWITCHER,
                'label_on'     => __( 'Show', 'yankee-toolkit' ),
                'label_off'    => __( 'Hide', 'yankee-toolkit' ),
                'return_value' => 'yes',
                'default'      => 'yes',
            ]
        );

        $this->end_controls_section();

        $this->start_controls_section(
            'widget_items_style',
            [
                'label' => esc_html__( 'Portfolio Items', 'yankee-toolkit' ),
                'tab'   => Controls_Manager::TAB_STYLE,
            ]
        );

        $this->add_responsive_control(
            'item_gap',
            [
                'label'      => esc_html__( 'Item Gap', 'yankee-toolkit' ),
                'type'       => Controls_Manager::SLIDER,
                'size_units' => ['px'],
                'range'      => [
                    'px' => [
                        'min'  => 0,
                        'max'  => 200,
                        'step' => 1,
                    ],
                ],
                'selectors'  => [
                    '{{WRAPPER}} .portfolio-list-items .portfolio-list-item:not(:last-child)' => 'margin-bottom: {{SIZE}}{{UNIT}}',
                ],
            ]
        );

        $this->add_control(
            'title_haeding',
            [
                'label'     => esc_html__( 'Title', 'yankee-toolkit' ),
                'type'      => Controls_Manager::HEADING,
                'separator' => 'before',
            ]
        );

        $this->add_control(
            'title_color',
            [
                'label'     => esc_html__( 'Color', 'yankee-toolkit' ),
                'type'      => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .portfolio-list-items .portfolio-list-item .title a' => 'color: {{VALUE}};',
                ],
            ]
        );

        $this->add_control(
            'title__hover_color',
            [
                'label'     => esc_html__( 'Hover Color', 'yankee-toolkit' ),
                'type'      => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .portfolio-list-items .portfolio-list-item .title a:hover' => 'color: {{VALUE}};',
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name'     => 'title_typography',
                'selector' => '{{WRAPPER}} .portfolio-list-items .portfolio-list-item .title',
            ]
        );

        $this->add_control(
            'categories_title',
            [
                'label'     => esc_html__( 'Categories', 'yankee-toolkit' ),
                'type'      => Controls_Manager::HEADING,
                'separator' => 'before',
                'condition' => [
                    'show_cat' => 'yes',
                ],
            ]
        );

        $this->add_control(
            'categories_color',
            [
                'label'     => esc_html__( 'Color', 'yankee-toolkit' ),
                'type'      => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .portfolio-list-items .portfolio-list-item .cat' => 'color: {{VALUE}};',
                ],
                'condition' => [
                    'show_cat' => 'yes',
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name'      => 'categories_typography',
                'selector'  => '{{WRAPPER}} .portfolio-list-items .portfolio-list-item .cat',
                'condition' => [
                    'show_cat' => 'yes',
                ],
            ]
        );

        $this->add_control(
            'desc_heading',
            [
                'label'     => esc_html__( 'Excerpt', 'yankee-toolkit' ),
                'type'      => Controls_Manager::HEADING,
                'separator' => 'before',
            ]
        );

        $this->add_control(
            'desc_color',
            [
                'label'     => esc_html__( 'Color', 'yankee-toolkit' ),
                'type'      => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .portfolio-list-items .portfolio-list-item .portfolio-desc p' => 'color: {{VALUE}};',
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name'     => 'desc_typography',
                'selector' => '{{WRAPPER}} .portfolio-list-items .portfolio-list-item .portfolio-desc p',
            ]
        );

        $this->add_control(
            'count_heading',
            [
                'label'     => esc_html__( 'Count', 'yankee-toolkit' ),
                'type'      => Controls_Manager::HEADING,
                'separator' => 'before',
            ]
        );

        $this->add_control(
            'count_color',
            [
                'label'     => esc_html__( 'Color', 'yankee-toolkit' ),
                'type'      => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .portfolio-list-items .portfolio-list-item .portfolio-desc .count' => 'color: {{VALUE}};',
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name'     => 'count_typography',
                'selector' => '{{WRAPPER}} .portfolio-list-items .portfolio-list-item .portfolio-desc .count',
            ]
        );

        $this->end_controls_section();

        $this->start_controls_section(
            'widget_button_style',
            [
                'label' => esc_html__( 'Button', 'yankee-toolkit' ),
                'tab'   => Controls_Manager::TAB_STYLE,
            ]
        );

        $this->add_control(
            'line_color',
            [
                'label'     => esc_html__( 'Line Color', 'yankee-toolkit' ),
                'type'      => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .portfolio-list-items .portfolio-list-item .line' => 'background-color: {{VALUE}};',
                ],
            ]
        );

        $this->add_responsive_control(
            'button_padding',
            [
                'label'      => esc_html__( 'Padding', 'yankee-toolkit' ),
                'type'       => Controls_Manager::DIMENSIONS,
                'size_units' => ['px', 'em', '%'],
                'selectors'  => [
                    '{{WRAPPER}} .portfolio-list-items .portfolio-list-item .portfolio-link' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->start_controls_tabs( 'button_tabs' );

        $this->start_controls_tab(
            'button_normal_tab',
            ['label' => esc_html__( 'Normal', 'yankee-toolkit' )]
        );

        $this->add_control(
            'button_normal_color',
            [
                'label'     => esc_html__( 'Color', 'yankee-toolkit' ),
                'type'      => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .portfolio-list-items .portfolio-list-item .portfolio-link' => 'color: {{VALUE}};',
                ],
            ]
        );

        $this->add_control(
            'button_normal_bg',
            [
                'label'     => esc_html__( 'Background', 'yankee-toolkit' ),
                'type'      => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .portfolio-list-items .portfolio-list-item .portfolio-link' => 'background-color: {{VALUE}};',
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Border::get_type(),
            [
                'name'     => 'button_normal_border',
                'label'    => esc_html__( 'Border', 'yankee-toolkit' ),
                'selector' => '{{WRAPPER}} .portfolio-list-items .portfolio-list-item .portfolio-link',
            ]
        );

        $this->add_control(
            'button_normal_border_radius',
            [
                'label'      => esc_html__( 'Border Radius', 'yankee-toolkit' ),
                'type'       => Controls_Manager::DIMENSIONS,
                'size_units' => ['px', 'em', '%'],
                'selectors'  => [
                    '{{WRAPPER}} .portfolio-list-items .portfolio-list-item .portfolio-link' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->end_controls_tab();

        $this->start_controls_tab(
            'button_hover_tab',
            ['label' => esc_html__( 'Hover', 'yankee-toolkit' )]
        );

        $this->add_control(
            'button_hover_color',
            [
                'label'     => esc_html__( 'Color', 'yankee-toolkit' ),
                'type'      => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .portfolio-list-items .portfolio-list-item .portfolio-link:hover' => 'color: {{VALUE}};',
                ],
            ]
        );

        $this->add_control(
            'button_hover_bg',
            [
                'label'     => esc_html__( 'Background', 'yankee-toolkit' ),
                'type'      => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .portfolio-list-items .portfolio-list-item .portfolio-link:hover' => 'Background-color: {{VALUE}};',
                ],
            ]
        );

        $this->add_control(
            'border_colr',
            [
                'label'     => esc_html__( 'Border Color', 'yankee-toolkit' ),
                'type'      => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .portfolio-list-items .portfolio-list-item .portfolio-link:hover' => 'border-color: {{VALUE}};',
                ],
            ]
        );

        $this->end_controls_tab();

        $this->end_controls_tabs();

        $this->end_controls_section();

    }

    protected function render() {
        $settings = $this->get_settings();
        $count = 0;
        ?>
            <div class="yankee-portfolio-widgets">
                <div class="portfolio-list-items">
                    <?php
                        if ( $settings['portfolio_cats'] ) {
                            $args = array(
                                'post_type'      => 'portfolios',
                                'post_status'    => 'publish',
                                'posts_per_page' => $settings['show_count'],
                                'orderby'        => $settings['order_by'],
                                'order'          => $settings['short_by'],
                                'ignore_sticky_posts' => 1,
                                'tax_query' => array(
                                    array(
                                        'taxonomy' => 'portfolios_cat',
                                        'field'    => 'slug',
                                        'terms'    => $settings['portfolio_cats'],
                                    ),
                                ),
                            );
                        } else {
                            $args = array(
                                'post_type'      => 'portfolios',
                                'post_status'    => 'publish',
                                'posts_per_page' => $settings['show_count'],
                                'orderby'        => $settings['order_by'],
                                'order'          => $settings['short_by'],
                                'ignore_sticky_posts' => 1,
                            );
                        }

                        $wp_query = new WP_Query($args);
                        while ( $wp_query -> have_posts() ) : $wp_query -> the_post();
                            $idd = get_the_ID();
                            $count++;

                            if ( $settings['title_word'] ) {
                                $the_title = wp_trim_words(get_the_title(), $settings['title_word']);
                            } else {
                                $the_title = get_the_title();
                            }

                            if ( $settings['content_word'] ) {
                                $the_content = wp_trim_words(get_the_content(), $settings['content_word']);
                            } else {
                                $the_content = get_the_excerpt();
                            }

                            $portfolio_categorys = get_the_terms( $idd, 'portfolios_cat' );
                            $item_cat_name = '';

                            if ($portfolio_categorys && ! is_wp_error( $portfolio_categorys) ){
                                $cat_list_name = array();

                                foreach ($portfolio_categorys as $category) {
                                    $cat_list_name[] = $category->name;
                                }

                                $item_cat_name = join( ", ", $cat_list_name );
                            }

                        ?>

                        <div class="portfolio-list-item">
                            <div class="row align-items-center">
                                <div class="col-lg-6 col-md-6">
                                    <div class="portfolio-thumb">
                                        <?php
                                            echo get_the_post_thumbnail( $idd, $settings['portfolio_thumbnail_size'] );
                                        ?>
                                    </div>
                                </div>
                                <div class="col-lg-6 col-md-6">
                                    <div class="portfolio-desc">
                                        <span class="count">
                                            <?php
                                                if( $count < 10 ) {
                                                    echo esc_html('0' . $count);
                                                } else {
                                                    echo esc_html($count);
                                                }
                                            ?>
                                        </span>
                                        <span class="line"></span>
                                        <h4 class="title"><a href="<?php echo esc_url( get_the_permalink() ) ?>"><?php echo esc_html( $the_title ); ?></a></h4>

                                        <?php if( !empty( $item_cat_name ) && $settings['show_cat'] ) : ?>
                                            <span class="cat"><?php echo esc_html( $item_cat_name ) ?></span>
                                        <?php endif; ?>

                                        <p><?php echo wp_kses_post( $the_content ) ?></p>
                                        <a href="<?php echo esc_url( get_the_permalink() ) ?>" class="portfolio-link"><?php echo esc_html( $settings['button_text'] ) ?></a>
                                    </div>
                                </div>
                            </div>
                        </div>

                        <?php endwhile;
                        wp_reset_postdata();
                    ?>
                </div>
            </div>
        <?php
    }

    protected function select_param_cates_post() {
        $category = get_terms( 'portfolios_cat' );
        $cat = [];
        foreach ( $category as $item ) {
            if ( $item ) {
                $cat[$item->slug] = $item->name;
            }
        }
        return $cat;
    }
}